/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */
namespace org.owasp.appsensor.intrusiondetection.reference
{
    using org.owasp.appsensor;
    using org.owasp.appsensor.intrusiondetection;
    using System;
    using System.Collections.Generic;
    using System.Net.Mail;
    using System.Net;
    using System.Net.Security;
    using System.Security.Cryptography.X509Certificates;
    using AppSensor2.configuration;

    /**
     * Reference implementation of the ResponseAction interface. This 
     * implementation handles the following response actions: 
     * 
     * <ul>
     * 		<li>"log" - logs the activity</li>
     *  	<li>"logout" - logs the currently logged in user out (if one exists)</li>
     *   	<li>"disable" - disables the account of the currently logged in user (if one exists)</li>
     *   	<li>"disableComponent" - disables access to the location of the intrusion using 
     *   		the AppSensorServiceController
     *   	</li> 
     *   	<li>"disableComponentForUser" - disables access to the location of the intrusion using 
     *   		the AppSensorServiceController for the currently logged in user(if one exists)
     *   	</li>    
     *   	<li>"smsAdmin" - sends sms to Administrator whose number is configured in appsensor.properties file
     *   	</li>   
     *   	<li>"emailAdmin" - sends email to Administrator whose address is configured in appsensor.properties file
     *   	</li>    
     * </ul>
     * 
     * The configuration point to specify another class that : the ResponseAction interface
     * is in the appsensor.properties file in the .esapi directory.
     * 
     * @author Michael Coates (michael.coates .at. owasp.org) 
     *         <a href="http://www.aspectsecurity.com">Aspect Security</a>
     * @author John Melton (jtmelton .at. gmail.com)
     *         <a href="http://www.jtmelton.com/">jtmelton</a>
     * @since February 24, 2010
     * @see org.owasp.appsensor.intrusiondetection.ResponseAction
     */
    public class DefaultResponseAction : ResponseAction
    {

        private static volatile ResponseAction singletonInstance;

        public static ResponseAction GetInstance()
        {
            if (singletonInstance == null)
            {
                lock (typeof(DefaultResponseAction))
                {
                    if (singletonInstance == null)
                    {
                        singletonInstance = new DefaultResponseAction();
                    }
                }
            }
            return singletonInstance;
        }

        private ASLogger logger = APPSENSOR.AsUtilities.GetLogger("DefaultResponseAction");

        private enum MailType { EMAIL, SMS }
        private static int SMS_MESSAGE_LENGTH = 160;

        /**
         * {@inheritDoc}
         */
        public bool HandleResponse(String action, AppSensorIntrusion currentIntrusion)
        {

            if (action != null)
            {
                //log issue ASR-A
                if (action.Equals("log", StringComparison.OrdinalIgnoreCase))
                {
                    return LogEvent(currentIntrusion);
                }

                //sms administrator (via email to sms account) to notify of what action has occurred ASR-B
                if (action.Equals("smsAdmin", StringComparison.OrdinalIgnoreCase))
                {
                    return SmsAdmin(currentIntrusion);
                }

                //pass action onto a proxy to handle e.g. honeypot or other hardware ASR-N
                if (action.Equals("proxy", StringComparison.OrdinalIgnoreCase))
                {
                    //return SmsAdmin(currentIntrusion);
                }

                //email administrator to notify of what action has occurred ASR-B
                if (action.Equals("emailAdmin", StringComparison.OrdinalIgnoreCase))
                {
                    return EmailAdmin(currentIntrusion);
                }

                //disable access to this component for all users ASR-I
                if (action.Equals("disableComponent", StringComparison.OrdinalIgnoreCase))
                {
                    return DisableComponent(currentIntrusion);
                }

                //can't perform the following actions if the user isn't logged in
                ASUser user = APPSENSOR.AsUtilities.CurrentUser;
                if (user.Anonymous)
                {
                    logger.Debug("Can't process this action as there is an anonymous user.");
                    return true;
                }

                //log the user out ASR-J
                if (action.Equals("logout", StringComparison.OrdinalIgnoreCase))
                {
                    return Logout(currentIntrusion.User);
                }

                //disable the user ASR-K
                if (action.Equals("disable", StringComparison.OrdinalIgnoreCase))
                {
                    return Disable(currentIntrusion.User);
                }

                //disable access to this component for a specific user ASR-I
                if (action.Equals("disableComponentForUser", StringComparison.OrdinalIgnoreCase))
                {
                    return DisableComponentForUser(currentIntrusion);
                }
            }

            //if we got here, it means someone has requested an unsupported action
            throw new InvalidOperationException("There has been a request for an action " +
                    "that is not supported by this response handler.  The requested action is: " + action);
        }

        /**
         * Log the current event
         * @param user currently logged in user
         * @param currentIntrusion current intrusion 
         * @return true if successful
         */
        private bool LogEvent(AppSensorIntrusion currentIntrusion)
        {
            ASUser user = currentIntrusion.User;
            logger.Debug("Response Action: Logging issue performed by " + user.AccountId);
            logger.Fatal("INTRUSION - Multiple intrusions observed by AppSensorUser:" + user.AccountName +
                    ", UserID:" + user.AccountId + ", Action: Logging Event with code: " + currentIntrusion.EventCode);
            return true;
        }

        /**
         * Logout the currently logged in user
         * @param user currently logged in user
         * @return true if successful
         */
        private bool Logout(ASUser user)
        {
            logger.Debug("Response Action: Logging Out User " + user.AccountId);
            logger.Fatal("INTRUSION - Multiple intrusions observed by AppSensorUser:" +
                    user.AccountName + ", UserID:" + user.AccountId + ", Action: Logging out malicious account");
            user.Logout();
            return true;
        }

        /**
         * Disable the currently logged in user
         * @param user currently logged in user
         * @return true if successful
         */
        private bool Disable(ASUser user)
        {
            logger.Debug("Response Action: Disabling User " + user.AccountId);
            logger.Fatal("INTRUSION - Multiple intrusions observed by AppSensorUser:" +
                    user.AccountName + ", UserID:" + user.AccountId + ", Action: Disabling and logging out malicious account");
            user.Disable();
            user.Logout();
            return true;
        }

        /**
         * Disables the function associated with the current intrusion for the 
         * designated period of time.  
         * @param currentIntrusion current intrusion
         * @return true if successful
         */
        private bool DisableComponent(AppSensorIntrusion currentIntrusion)
        {
            //retrieve threshold from config file
            AppSensorThreshold threshold = AppSensorIntrusionDetector.GetAppSensorThreshold(currentIntrusion.EventCode);

            //determine service to disable based on current intrusion
            String serviceToDisable = currentIntrusion.Location;

            if (threshold != null)
            {
                //get amount of time to disable component for
                int duration = threshold.disableComponentDuration;
                var timeScale = threshold.disableComponentTimeScale;

                //actually disable service for all users
                AppSensorServiceController.DisableService(serviceToDisable, duration, timeScale);
                logger.Warning("Successfully disabled service: Suspending Service <" + serviceToDisable +
                        "> for <" + duration + "> <" + timeScale + ">");
                return true;
            }
            else
            {
                logger.Warning("Could not disabled service <" + serviceToDisable + "> because" +
                        " there is no configuration setup in the Esapi.properties file");
                return false;
            }
        }

        /**
         * Disables the function associated with the current intrusion for the 
         * designated period of time, but only for the specified user.
         * @param currentIntrusion current intrusion
         * @return true if successful
         */
        private bool DisableComponentForUser(AppSensorIntrusion currentIntrusion)
        {
            //retrieve threshold from config file
            AppSensorThreshold threshold = AppSensorIntrusionDetector.GetAppSensorThreshold(currentIntrusion.EventCode);

            //determine service to disable based on current intrusion
            String serviceToDisable = currentIntrusion.Location;

            //retrieve user from intrusion
            ASUser user = currentIntrusion.User;
            //get userid of user
            String userId = user.AccountId.ToString();

            if (threshold != null)
            {
                //get amount of time to disable component for
                var duration = threshold.disableComponentForUserDuration;
                var timeScale = threshold.disableComponentForUserTimeScale;

                //actually disable service for user
                AppSensorServiceController.DisableServiceForUser(serviceToDisable, userId, duration, timeScale);
                logger.Warning("Successfully disabled service: Suspending Service <" + serviceToDisable +
                        "> for userId/username <" + userId + "/" + user.AccountName + "> for <" + duration + "> <" + timeScale + ">");
                return true;
            }
            else
            {
                logger.Warning("Could not disabled service <" + serviceToDisable + "> for " +
                        "userId/username <" + userId + "/" + user.AccountName + "> because" +
                        " there is no configuration setup in the Esapi.properties file");
                return false;
            }
        }

        /**
         * SMS administrator (via email to sms account) to notify of what action has occurred
         * @param currentIntrusion current intrusion 
         * @return true if successful
         */
        private bool SmsAdmin(AppSensorIntrusion currentIntrusion)
        {
            ASUser user = currentIntrusion.User;
            logger.Debug("Response Action: Sending SMS to Admin due to issue performed by " + user.AccountId);
            logger.Fatal("INTRUSION - Multiple intrusions observed by AppSensorUser:" + user.AccountName +
                    ", UserID:" + user.AccountId + ", Action: Sending SMS to Admin with code: " + currentIntrusion.EventCode);
            String subject = "AppSensor Issue Detection";
            String body = "User:" + user.AccountName + " (ID:" + user.AccountId + ") " +
                    "/ caused event code: " + currentIntrusion.EventCode;

            //truncate message for sms transmission if necessary
            int totalLength = subject.Length + body.Length;
            if (totalLength > SMS_MESSAGE_LENGTH)
            {
                //get leftover length we have to use for the body
                int bodyLength = SMS_MESSAGE_LENGTH - totalLength;
                body = body.Substring(0, bodyLength - 1);
            }

            return SendEmail(MailType.SMS, subject, body);
        }

        /**
         * Email administrator to notify of what action has occurred
         * @param currentIntrusion current intrusion 
         * @return true if successful
         */
        private bool EmailAdmin(AppSensorIntrusion currentIntrusion)
        {
            ASUser user = currentIntrusion.User;
            logger.Debug("Response Action: Sending email to Admin due to issue performed by " + user.AccountId);
            logger.Fatal("INTRUSION - Multiple intrusions observed by AppSensorUser:" + user.AccountName +
                    ", UserID:" + user.AccountId + ", Action: Sending email to Admin with code: " + currentIntrusion.EventCode);
            String subject = "AppSensor Issue Detection";
            String body = "User:" + user.AccountName + " (ID:" + user.AccountId + ") " +
                    "/ caused event code: " + currentIntrusion.EventCode;

            body += "\r\n" + "See http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project#tab=Detection_Points for more information.";

            return SendEmail(MailType.EMAIL, subject, body);
        }

        /**
         * This method is used for sending email to cover both standard email 
         * as well as SMS
         * @param type SMS or standard email
         * @param subject subject of email
         * @param body body of email
         * @return bool indicating true if email was sent successfully, false if not
         */
        private bool SendEmail(MailType type, String subject, String body)
        {
            AppSensorSecurityConfiguration assc = (AppSensorSecurityConfiguration)AppSensorSecurityConfiguration.GetInstance();
            try
            {
                String protocol;
                String host;
                int port;
                bool isAuthenticationRequired = false;
                bool isProtocolSecure = false;
                String user;
                String password;
                String from;
                IEnumerable<String> toAddresses;

                if (MailType.EMAIL.Equals(type))
                {
                    protocol = assc.EmailAdmin.Protocol;
                    host = assc.EmailAdmin.Host;
                    port = assc.EmailAdmin.Port;
                    isAuthenticationRequired = assc.EmailAdmin.AuthenticationRequired;
                    user = assc.EmailAdmin.AuthenticationUser;
                    password = assc.EmailAdmin.AuthenticationPassword;
                    from = assc.EmailAdmin.FromAccount;
                    toAddresses = assc.EmailAdmin.ToAccounts.ToList();
                }
                else if (MailType.SMS.Equals(type))
                {
                    //protocol = assc.GetSMSEmailProtocol();
                    //host = assc.GetSMSEmailHost();
                    //port = assc.GetSMSEmailPort();
                    //isAuthenticationRequired = assc.GetSMSEmailAuthenticationRequired();
                    //user = assc.GetSMSEmailAuthenticationUser();
                    //password = assc.GetSMSEmailAuthenticationPassword();
                    //from = assc.GetSMSEmailFromAccount();
                    //toAddresses = assc.GetSMSEmailToAccounts();
                    return true; // Not available yet
                }
                else
                {
                    throw new InvalidOperationException("Invalid MailType passed in");
                }

                if ("smtps".Equals(protocol))
                {
                    isProtocolSecure = true;
                    //reset protocol back to smtp to make it work properly, but 
                    //do everything over https below
                    protocol = "smtp";
                }

                ServicePointManager.ServerCertificateValidationCallback = new RemoteCertificateValidationCallback(ValidateServerCertificate);
                SmtpClient client = new SmtpClient();
                client.DeliveryMethod = SmtpDeliveryMethod.Network;
                client.EnableSsl = true;
                client.Host = host;
                client.Port = port;

                // setup Smtp authentication
                System.Net.NetworkCredential credentials = new System.Net.NetworkCredential(user, password);
                client.UseDefaultCredentials = false;
                client.Credentials = credentials;

                MailMessage msg = new MailMessage();
                msg.From = new MailAddress(from);
                msg.ReplyToList.Add(new MailAddress("no-reply@pixelpin.co.uk"));
                foreach (String toAddress in toAddresses)
                {
                    msg.To.Add(new MailAddress(toAddress));
                }

                msg.Subject = subject;
                msg.IsBodyHtml = false;
                msg.Body = body;

                try
                {
                    client.Send(msg);
                    logger.Info("Email sent to: ");
                    return true;
                }
                catch (Exception ex)
                {
                    logger.Error("Error occurred sending email message: ", ex);
                    return false;
                }
                return true;
            }
            catch (Exception e)
            {
                String mtype;
                if (MailType.SMS.Equals(type))
                {
                    mtype = "SMS";
                }
                else
                {
                    mtype = "standard email";
                }
                logger.Fatal("Could not send email (type - " + mtype + ") to admin: ", e);

                return false;
            }

        }

        private static bool ValidateServerCertificate(object sender,
            X509Certificate certificate,
            X509Chain chain,
            SslPolicyErrors sslPolicyErrors)
        {
            // TODO - decide correct verification policy, particularly related to man-in-middle attacks
            if (sslPolicyErrors == SslPolicyErrors.None)
                return true;
            else
                return false;
        }
    }
}